import { Navbar } from "@/components/navbar"
import { Footer } from "@/components/footer"
import { ChatInterface } from "@/components/chat-interface"
import { Button } from "@/components/ui/button"
import { Card, CardContent } from "@/components/ui/card"
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Users, Video, FileText, Share2, ArrowLeft } from "lucide-react"
import Image from "next/image"
import Link from "next/link"

export default function StudyRoom({ params }: { params: { id: string } }) {
  // This would be fetched from a database in a real application
  const roomData = {
    id: params.id,
    title: "Advanced Calculus Study Group",
    subject: "Mathematics",
    description:
      "In this session, we'll be covering integration techniques, including substitution, integration by parts, and partial fractions. We'll also work through practice problems from the textbook chapters 7-9.",
    participants: [
      { name: "Prof. David Chen", role: "Host", avatar: "/images/avatar-prof.jpg", online: true },
      { name: "Sarah Johnson", role: "Student", avatar: "/images/avatar-1.jpg", online: true },
      { name: "Miguel Rodriguez", role: "Student", avatar: "/images/avatar-2.jpg", online: true },
      { name: "Emma Lewis", role: "Student", avatar: "/images/avatar-5.jpg", online: false },
      { name: "Carlos Martinez", role: "Student", avatar: "/images/avatar-6.jpg", online: true },
    ],
    resources: [
      { name: "Calculus Textbook Ch. 7-9", type: "PDF", size: "4.2 MB" },
      { name: "Integration Techniques Cheat Sheet", type: "PDF", size: "1.8 MB" },
      { name: "Practice Problems Set", type: "PDF", size: "2.5 MB" },
    ],
    images: ["/images/calculus-whiteboard-1.jpg", "/images/calculus-whiteboard-2.jpg", "/images/calculus-notes.jpg"],
  }

  return (
    <div className="min-h-screen flex flex-col">
      <Navbar />

      <main className="flex-grow">
        <div className="bg-teal-500 text-white py-6">
          <div className="container mx-auto px-4">
            <Link href="/live-study" className="inline-flex items-center text-white hover:underline mb-4">
              <ArrowLeft className="h-4 w-4 mr-1" />
              Back to Study Rooms
            </Link>
            <h1 className="text-2xl md:text-3xl font-bold">{roomData.title}</h1>
            <p className="text-lg">{roomData.subject}</p>
          </div>
        </div>

        <div className="container mx-auto px-4 py-8">
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
            <div className="lg:col-span-2">
              <Card className="mb-8">
                <CardContent className="p-0">
                  <div className="relative h-64 md:h-96 bg-gray-100">
                    <div className="absolute inset-0 flex items-center justify-center">
                      <div className="text-center">
                        <Image
                          src="/images/virtual-classroom-placeholder.jpg"
                          alt="Virtual Classroom"
                          width={600}
                          height={400}
                          className="mx-auto rounded-lg shadow-lg"
                        />
                        <div className="mt-4">
                          <Button className="bg-teal-500 hover:bg-teal-600">
                            <Video className="h-4 w-4 mr-2" />
                            Join Video Session
                          </Button>
                        </div>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <ChatInterface roomId={params.id} />
            </div>

            <div>
              <Card className="mb-6">
                <CardContent className="p-6">
                  <h3 className="text-lg font-bold mb-2">About this Study Room</h3>
                  <p className="text-gray-600 mb-4">{roomData.description}</p>

                  <div className="flex items-center justify-between mb-4">
                    <h4 className="font-medium">Participants ({roomData.participants.length})</h4>
                    <Button variant="ghost" size="sm" className="text-teal-600">
                      <Users className="h-4 w-4 mr-1" />
                      Invite
                    </Button>
                  </div>

                  <div className="space-y-3 mb-6">
                    {roomData.participants.map((participant, index) => (
                      <div key={index} className="flex items-center justify-between">
                        <div className="flex items-center">
                          <Avatar className="h-8 w-8 mr-2">
                            <AvatarImage src={participant.avatar || "/placeholder.svg"} alt={participant.name} />
                            <AvatarFallback>{participant.name.charAt(0)}</AvatarFallback>
                          </Avatar>
                          <div>
                            <div className="text-sm font-medium">{participant.name}</div>
                            <div className="text-xs text-gray-500">{participant.role}</div>
                          </div>
                        </div>
                        <div
                          className={`h-2 w-2 rounded-full ${participant.online ? "bg-green-500" : "bg-gray-300"}`}
                        />
                      </div>
                    ))}
                  </div>

                  <div className="border-t pt-4">
                    <h4 className="font-medium mb-3">Resources</h4>
                    <div className="space-y-2">
                      {roomData.resources.map((resource, index) => (
                        <div key={index} className="flex items-center justify-between p-2 bg-gray-50 rounded">
                          <div className="flex items-center">
                            <FileText className="h-4 w-4 text-gray-500 mr-2" />
                            <div>
                              <div className="text-sm font-medium">{resource.name}</div>
                              <div className="text-xs text-gray-500">
                                {resource.type} • {resource.size}
                              </div>
                            </div>
                          </div>
                          <Button variant="ghost" size="sm">
                            Download
                          </Button>
                        </div>
                      ))}
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardContent className="p-6">
                  <h3 className="text-lg font-bold mb-4">Shared Images</h3>
                  <Tabs defaultValue="images">
                    <TabsList className="mb-4">
                      <TabsTrigger value="images">Images</TabsTrigger>
                      <TabsTrigger value="whiteboard">Whiteboard</TabsTrigger>
                    </TabsList>
                    <TabsContent value="images">
                      <div className="grid grid-cols-2 gap-2">
                        {roomData.images.map((image, index) => (
                          <div key={index} className="relative aspect-square rounded-md overflow-hidden">
                            <Image
                              src={image || "/placeholder.svg"}
                              alt={`Shared image ${index + 1}`}
                              fill
                              className="object-cover"
                            />
                          </div>
                        ))}
                        <div className="relative aspect-square rounded-md overflow-hidden bg-gray-100 flex items-center justify-center border-2 border-dashed border-gray-300">
                          <Button variant="ghost" className="text-gray-500">
                            + Add Image
                          </Button>
                        </div>
                      </div>
                    </TabsContent>
                    <TabsContent value="whiteboard">
                      <div className="bg-gray-100 rounded-md p-4 text-center h-48 flex items-center justify-center">
                        <Button className="bg-teal-500 hover:bg-teal-600">Open Collaborative Whiteboard</Button>
                      </div>
                    </TabsContent>
                  </Tabs>

                  <div className="mt-4">
                    <Button variant="outline" className="w-full">
                      <Share2 className="h-4 w-4 mr-2" />
                      Share Room Link
                    </Button>
                  </div>
                </CardContent>
              </Card>
            </div>
          </div>
        </div>
      </main>

      <Footer />
    </div>
  )
}
